/**
 * \file: signature_db_writer.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * \component: authorization level daemon
 *
 * \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
 *
 * \copyright (c) 2010, 2011 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/
#include <limits.h>
#include <stdio.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/stat.h>
#include <string.h>
#include <errno.h>
#include <arpa/inet.h>

#include "signature_db_writer.h"

#include "model/signature_db.h"
#include "encryption/signature.h"

//--------------------------------------- private attributes ----------------------------------------------------------
static int db_fd=-1;
static const char *db_writer_sysroot_dir="";
static EVP_PKEY *privkey=NULL;
//---------------------------------------------------------------------------------------------------------------------

//--------------------------------------- public function definition --------------------------------------------------
error_code_t signature_db_writer_init(const char *privkey_path)
{
	error_code_t result;

	result=signature_init();
	if (result!=RESULT_OK)
		printf("ERROR: Initializing openssl signature algorithms failed.\n");
	else
		result=signature_read_private_key(privkey_path,&privkey);

	if (result==RESULT_FILE_NOT_FOUND)
		printf("ERROR: Unable to read private key from file: %s\n",privkey_path);
	else if (result!=RESULT_OK)
		printf("ERROR: Private key invalid in file: %s\n",privkey_path);

	return result;
}

void signature_db_writer_deinit(void)
{
	if (privkey!=NULL)
	{
		signature_destroy_key(privkey);
		privkey=NULL;
	}

	signature_deinit();
}

error_code_t signature_db_writer_create_db(const char *sysroot_dir, const char *db_file_path)
{
	char path[PATH_MAX];

	if (db_fd!=-1)
		signature_db_writer_close();

	db_writer_sysroot_dir=sysroot_dir;

	//opening the data base
	snprintf(path, PATH_MAX,"%s%s",sysroot_dir,db_file_path);
	(void)unlink(path);
	db_fd=open(path,O_CREAT|O_WRONLY,0666);
	if (db_fd==-1)
	{
		printf("ERROR: Unable to create the data base file \'%s\': %s\n",path,strerror(errno));
		return RESULT_SIGNATURE_DB_CREATION_FAILED;
	}

	return RESULT_OK;
}

error_code_t signature_db_writer_add_from_file(const char *path)
{
	error_code_t result;
	struct stat stat_result;
	serialized_signature_db_entry_t new_entry;
	char abs_path[PATH_MAX];
	uint32_t path_len;

	//did someone create a db file for us?
	if (db_fd==-1)
		return RESULT_INVALID;

	//bringing sysroot into the game and checking for the presence of the file
	snprintf(abs_path,PATH_MAX,"%s%s",db_writer_sysroot_dir,path);
	if (stat(abs_path, &stat_result)==-1)
		return RESULT_FILE_NOT_FOUND;

	result=signature_create_for_file(abs_path, new_entry.signature, privkey);
	if (result!=RESULT_OK)
	{
		printf("ERROR: Signing of script \'%s\' failed.\n",abs_path);
		return result;
	}
	path_len=strlen(path)+1;
	new_entry.path_len=htonl(path_len);

	if (write(db_fd,&new_entry,sizeof(serialized_signature_db_entry_t))!=(ssize_t)sizeof(serialized_signature_db_entry_t))
		result=RESULT_SIGNATURE_DB_CREATION_FAILED;
	if (write(db_fd,path,path_len)!=(ssize_t)path_len)
		result=RESULT_SIGNATURE_DB_CREATION_FAILED;

	return result;
}

void signature_db_writer_close(void)
{
	if (db_fd!=-1)
	{
		close(db_fd);
		db_fd=-1;
	}
}

error_code_t signature_db_writer_sign_db(const char *sysroot, const char *sig_db_path)
{
	char abs_path[PATH_MAX];
	char signature[RESPONSE_SIGNATURE_SIZE_USED];
	error_code_t result;

	if (privkey==NULL)
		return RESULT_INVALID;

	//creating the signature of the data base
	snprintf(abs_path,PATH_MAX,"%s%s",sysroot,sig_db_path);
	result=signature_create_for_file(abs_path, signature,privkey);

	//writing it down
	if (result==RESULT_OK)
	{
		snprintf(abs_path,PATH_MAX,"%s%s%s",sysroot,sig_db_path,SIGNATURE_EXT);
		result=signature_write_to_file(abs_path,signature);
	}
	else
		printf("ERROR: Unable to write down the signature into \'%s\'.",abs_path);

	return result;
}
//---------------------------------------------------------------------------------------------------------------------
